function [sigfilter,n] = filter_3sFIR(sig,f,a,dev,fs)
% A three-stage FIR filter processing 
% using Parks-McClellan optimal equiripple FIR order estimator
%
% The last 4 input arguments are identical to those used in FIRPMORD
%  sig: is the signal to be filtered arranged, each row is a measurement
%    f: a vector of cutoff frequencies in Hz, in ascending order between 
%       0 and half the sampling frequency fs. The first frequency band 
%       always starts at zero, and the last always ends at fs/2.
%    a: a vector specifying the desired function's amplitude on the bands 
%       defined by F. The length of F is twice the length of A, minus 2 
%       (it must therefore be even).
%  dev: a vector of maximum deviations or ripples (in linear units) 
%       allowable for each band. dev must have the same length as A.
%   fs: the sampling frequency.
%
%  output n is the order of FIR filter.
%
%  Details of input arguments and examples, please help FIRPMORD.
%
%  [Author]: Lu, Chia-Feng 2013.11.08
%
%  EXAMPLE of using FIRPMORD
%    Design a lowpass filter with a passband-edge frequency of 1500Hz, a 
%    stopband-edge of 2000Hz, passband ripple of 0.01, stopband ripple 
%    of 0.1, and a sampling frequency of 8000Hz:
%
%        [n,fo,mo,w] = firpmord([1500 2000], [1 0], [0.01 0.1], 8000 );

[n,fo,mo,w] = firpmord(f, a, dev, fs);
b = firpm(n,fo,mo,w);

if size(sig,2)<=3*n
    error('Signal must have length more than 3 times filter order for FILTFILT process')
end

sigfilter=[];
for i=1:size(sig,1)
    sigfilter(i,:) = filtfilt(b,1,sig(i,:));
end
