function writex(varargin)
%save images as either .sdt or .nii filetypes 
%input: writex(img,'filename.ext',precision)
%           img - the image to save
%           filename - the name of the file to save as
%                     ext can be <sdt> or <nii>
%           precision - the datatype, can be int16, int32, float32
%                       if nifti it can also be uint8, complex64, float64,
%                         RGB24, int8, uint16, uint32, complex128
%       writex(img,'filename',precision,type)
%           type - the file extension if not given in filename,
%                  should be 'sdt' or 'nii'
%       writex(img,'filename',precision,type,info)
%           if saving as nifti, info can be:
%             1. header info struct from readx from nifti or dicom to 
%                convert relevant stuff to .nii header
%             2. the voxel size in an array, [dim1 dim2 dim3]
%if readx was used to load .sdt or .nii data, then the returned info2 data
%   can be used as inputs for writex to save in the same precision and 
%   ie: >> [img,info,info2]=readx('file.nii')
%       >> writex(img,'file2',info2.precision,info2.filetype)
%if the extension/type is not given at in either the filename or type, 
%   the default is given below as default='sdt' or default='nii'
default='nii';

[filepath filename type]=fileparts(varargin{2});
filename=[filepath filesep filename];
precision=varargin{3};
switch nargin
    case 3
        if ~isempty(type)
            type=type(2:end);
        else
            type=default;
        end
        info=[];
    case 4
        type=varargin{4};
        info=[];
    case 5
        type=varargin{4};
        info=varargin{5};
end

if ~(strcmp(type,'sdt')||strcmp(type,'nii')||strcmp(type,'spr'))
    disp(['Warning: File type ''' type ''' is not valid. Using default, ''' default ''''])
    type=default;
end
switch type
    case {'sdt','spr'}
        writexsdt(varargin{1},precision,filename);
    case 'nii'
        writexnii(varargin{1},precision,filename,info);
end

end

function writexsdt(img,precision,fname)
%function writesdt(img,precision,fname);
% Qiang Shen, CCNI, UMMS

fclose('all');
f1=fopen(strcat(fname,'.sdt'),'w','l');
f2=fopen(strcat(fname,'.spr'),'w');
%fwrite(f1,img,precision);

switch precision
case 'int32'
    byt = 32/8;
case 'int16'
    byt = 16/8;
case 'char'
    byt = 8/8;
case 'float32'
    byt = 32/8;
case 'single'
    byt = 32/8;
    precision='float32';
end

dim=ndims(img);
count2 = fwrite(f1,img,precision);
count = count2;
point = ftell(f1);
while point/byt~=length(img(:))
    fseek(f1,byt*floor(point/byt),'bof');
    count2 = fwrite(f1,img(floor(point/byt)+1:end),precision);
    count = count+count2;
    point = ftell(f1);
end
[message,errnum] = ferror(f1);

fprintf(f2,'numDim: %d\n',dim);
switch dim
case 2
    fprintf(f2,'dim: %3d %3d\n',size(img));
case 3
    fprintf(f2,'dim: %3d %3d %3d\n',size(img));
case 4
    fprintf(f2,'dim: %3d %3d %3d %3d\n',size(img));
case 5
    fprintf(f2,'dim: %3d %3d %3d %3d %3d\n',size(img));
end
switch precision
case 'int32'
    dataType='LWORD';
case 'int16'
    dataType='WORD';
case 'char'
    dataType='BYTE';
case 'float32'
    dataType='REAL';
end
fprintf(f2,'dataType: %s\n',dataType);
fprintf(f2,'sdtOrient:ax\n');
fclose(f1);
fclose(f2);
end

function writexnii(img,precis,fname,info)
switch precis
    case 'uint8',      precision = 2;
    case 'int16',      precision = 4;
    case 'int32',      precision = 8;
    case 'float32',    precision = 16;
    case 'float64',    precision = 64;
    case 'int8',       precision = 256;
    case 'uint16',     precision = 512;
    case 'uint32',     precision = 768;
    case 'complex64',  precision = 32;
    case 'RGB24',      precision = 128;
    case 'complex128', precision = 1792;
end
% create nifti image
if ~isempty(info)&&~isstruct(info)
    voxsize=info;
else voxsize=[1 1 1];
end
img_nii = make_nii(img, voxsize, [], precision);
if ~isempty(info)&&isstruct(info)
    if all(isfield(info,{'hk' 'dime' 'hist'})) %check if nifti header
        img_nii.hdr.hk=info.hk; img_nii.hdr.hist=info.hist; 
        img_nii.hdr.dime.pixdim=info.dime.pixdim;
        img_nii.hdr.dime.scl_slope=info.dime.scl_slope;
        img_nii.hdr.dime.scl_inter=info.dime.scl_inter;
        img_nii.hdr.dime.xyzt_units=info.dime.xyzt_units;
    elseif isfield(info,'Format')
    if strcmp(info.Format,'DICOM') %check if dicom header
        if strcmp(info.Manufacturer,'Philips Medical Systems')
            scale1=info.PerFrameFunctionalGroupsSequence.Item_1.PixelValueTransformationSequence.Item_1.RescaleSlope; 
            scale2=info.PerFrameFunctionalGroupsSequence.Item_1.PixelValueTransformationSequence.Item_1.RescaleIntercept;
            scale3=info.PerFrameFunctionalGroupsSequence.Item_1.Private_2005_140f.Item_1.Private_2005_100e;
            %%%real=( dicom*scale1+scale2 ) / ( scale1*scale3 );
            img_nii.hdr.dime.scl_slope=1/scale3;
            img_nii.hdr.dime.scl_inter=scale2/(scale1*scale3);
            pixdim=img_nii.hdr.dime.pixdim;
            pixdim(2:3)=info.PerFrameFunctionalGroupsSequence.Item_1.PixelMeasuresSequence.Item_1.PixelSpacing;
            pixdim(4)=info.PerFrameFunctionalGroupsSequence.Item_1.Private_2005_140f.Item_1.SliceThickness;
            img_nii.hdr.dime.pixdim=pixdim;
            %img_nii.dime.slice_duration=.001*info.PerFrameFunctionalGroupsSequence.Item_1.FrameContentSequence.Item_1.FrameAcquisitionDuration;
        end
        
    end
    end
end
% save nifti image to .nii file
save_nii(img_nii, [fname '.nii']);
end